import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
import time

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

[Frozen Code Start]

[Frozen Code End]

def run_skeleton_task():
    '''Generic skeleton for running the oracle plan in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video recording wrappers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve all object positions
        positions = get_object_positions()
        
        # Define quaternions for zero and ninety degree rotations around the Z‐axis
        zero_quat   = R.from_euler('z', 0,  degrees=True).as_quat()
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # === Oracle Plan Execution ===
        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan Step 1] rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("Task ended early during rotate.")
            return

        # Step 2: move to side position of middle drawer
        print("[Plan Step 2] move to side position of middle drawer")
        side_middle_pos = positions['middle_side_pos']
        obs, reward, done = move(env, task, side_middle_pos)
        if done:
            print("Task ended early during move-to-side.")
            return

        # Step 3: move to anchor position of middle drawer
        print("[Plan Step 3] move to anchor position of middle drawer")
        anchor_middle_pos = positions['middle_anchor_pos']
        obs, reward, done = move(env, task, anchor_middle_pos)
        if done:
            print("Task ended early during move-to-anchor.")
            return

        # Step 4: pick-drawer (grasp the drawer handle)
        print("[Plan Step 4] pick-drawer (grasp handle)")
        obs, reward, done = pick(env, task, anchor_middle_pos, approach_axis='x')
        if done:
            print("Task ended early during pick-drawer.")
            return

        # Step 5: pull to open the drawer
        print("[Plan Step 5] pull drawer to open")
        # Assume pulling 0.1m along the X axis opens the drawer
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("Task ended early during pull.")
            return

        # Step 6: pick tomato1 from the table
        print("[Plan Step 6] pick tomato1 from table")
        tomato1_pos = positions['tomato1']
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("Task ended early during pick tomato1.")
            return

        # Step 7: place tomato1 onto the plate
        print("[Plan Step 7] place tomato1 onto plate")
        plate_pos = positions['plate']
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("Task ended early during place tomato1.")
            return

        # Step 8: pick tomato2 from the table
        print("[Plan Step 8] pick tomato2 from table")
        tomato2_pos = positions['tomato2']
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("Task ended early during pick tomato2.")
            return

        # Step 9: place tomato2 onto the plate
        print("[Plan Step 9] place tomato2 onto plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("Task ended early during place tomato2.")
            return

        # Final check
        if not done:
            print("===== Task completed successfully! Final reward:", reward)
        else:
            print("===== Task ended with done=True.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()