# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

[Frozen Code Start]

[Frozen Code End]

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Execute Oracle Plan ===

        # Step 1: Pull open the bottom drawer (it's the only unlocked one)
        handle_pos = positions.get('bottom_anchor_pos')
        if handle_pos is None:
            raise KeyError("bottom_anchor_pos not found in object positions")
        print("[Task] Step 1: Pull open bottom drawer")
        # we assume gripper is already at handle position or pull will move relative
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Terminated after pull. Reward:", reward)
            return

        # Step 2: Pick up tomato1 from the table
        tomato1_pos = positions.get('tomato1')
        if tomato1_pos is None:
            raise KeyError("tomato1 not found in object positions")
        print("[Task] Step 2: Picking tomato1 at:", tomato1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(tomato1_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated after picking tomato1. Reward:", reward)
            return

        # Step 3: Place tomato1 onto the plate
        plate_pos = positions.get('plate')
        if plate_pos is None:
            raise KeyError("plate not found in object positions")
        print("[Task] Step 3: Placing tomato1 onto plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(plate_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated after placing tomato1. Reward:", reward)
            return

        # Step 4: Pick up tomato2 from the table
        tomato2_pos = positions.get('tomato2')
        if tomato2_pos is None:
            raise KeyError("tomato2 not found in object positions")
        print("[Task] Step 4: Picking tomato2 at:", tomato2_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(tomato2_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated after picking tomato2. Reward:", reward)
            return

        # Step 5: Place tomato2 onto the plate
        print("[Task] Step 5: Placing tomato2 onto plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(plate_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated after placing tomato2. Reward:", reward)
            return

        print("[Task] All steps completed successfully. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()