from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper to ninety_deg")
        ninety_quat = positions.get('ninety_deg')
        if ninety_quat is None:
            raise KeyError("ninety_deg quaternion not found in positions")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Task ended after rotation.")
            return

        # Step 2: Move to bottom side position
        print("[Task] Step 2: Moving to bottom_side_pos")
        bottom_side = positions.get('bottom_side_pos')
        if bottom_side is None:
            raise KeyError("bottom_side_pos not found in positions")
        obs, reward, done = move(env, task, target_pos=np.array(bottom_side))
        if done:
            print("[Task] Task ended after moving to bottom side.")
            return

        # Step 3: Move to bottom anchor position
        print("[Task] Step 3: Moving to bottom_anchor_pos")
        bottom_anchor = positions.get('bottom_anchor_pos')
        if bottom_anchor is None:
            raise KeyError("bottom_anchor_pos not found in positions")
        obs, reward, done = move(env, task, target_pos=np.array(bottom_anchor))
        if done:
            print("[Task] Task ended after moving to bottom anchor.")
            return

        # Step 4: Pull drawer open
        print("[Task] Step 4: Pulling the drawer open")
        # Assuming drawer handle pull along x-axis by 0.1m
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Task ended after pull.")
            return

        # Step 5: Pick first tomato (item1) from table
        print("[Task] Step 5: Picking item1 (first tomato)")
        item1_pos = positions.get('item1')
        if item1_pos is None:
            raise KeyError("item1 not found in positions")
        obs, reward, done = pick(env, task, target_pos=np.array(item1_pos))
        if done:
            print("[Task] Task ended after picking item1.")
            return

        # Step 6: Place first tomato onto plate
        print("[Task] Step 6: Placing item1 onto plate")
        plate_pos = positions.get('plate')
        if plate_pos is None:
            raise KeyError("plate not found in positions")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Task] Task ended after placing item1.")
            return

        # Step 7: Pick second tomato (item2) from table
        print("[Task] Step 7: Picking item2 (second tomato)")
        item2_pos = positions.get('item2')
        if item2_pos is None:
            raise KeyError("item2 not found in positions")
        obs, reward, done = pick(env, task, target_pos=np.array(item2_pos))
        if done:
            print("[Task] Task ended after picking item2.")
            return

        # Step 8: Place second tomato onto plate
        print("[Task] Step 8: Placing item2 onto plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Task] Task completed successfully with reward:", reward)
        else:
            print("[Task] Final step executed. Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()