import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper from zero to ninety degrees about Z
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        print("[Task] Step 1: rotate gripper to 90° about Z")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended after rotation.")
            return

        # Step 2: Move to the side position of the bottom drawer
        side_pos_bottom = positions['bottom_side_pos']
        print(f"[Task] Step 2: move to side_pos_bottom {side_pos_bottom}")
        obs, reward, done = move(env, task, side_pos_bottom)
        if done:
            print("[Task] Ended after move to side_pos_bottom.")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        anchor_pos_bottom = positions['bottom_anchor_pos']
        print(f"[Task] Step 3: move to anchor_pos_bottom {anchor_pos_bottom}")
        obs, reward, done = move(env, task, anchor_pos_bottom)
        if done:
            print("[Task] Ended after move to anchor_pos_bottom.")
            return

        # Step 4: Pull the bottom drawer open
        # Assumes pulling along the local X axis by 0.1m
        print("[Task] Step 4: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ended after pull.")
            return

        # Step 5: Pick the first tomato (item1) from the table
        item1_pos = positions['item1']
        print(f"[Task] Step 5: pick item1 at {item1_pos}")
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Ended after picking item1.")
            return

        # Step 6: Place the first tomato onto the plate
        plate_pos = positions['plate']
        print(f"[Task] Step 6: place item1 on plate at {plate_pos}")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ended after placing item1.")
            return

        # Step 7: Pick the second tomato (item2) from the table
        item2_pos = positions['item2']
        print(f"[Task] Step 7: pick item2 at {item2_pos}")
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Ended after picking item2.")
            return

        # Step 8: Place the second tomato onto the plate
        print(f"[Task] Step 8: place item2 on plate at {plate_pos}")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ended after placing item2.")
            return

        print("[Task] Oracle plan completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()