[Frozen Code Start]

[Frozen Code End]

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap task.step and task.get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions and Orientations ===
        positions = get_object_positions()
        # We expect get_object_positions to return:
        #   - 3D positions for keys ending in '_pos'
        #   - quaternions (as length-4 arrays) for keys 'zero_deg' and 'ninety_deg'
        zero_quat = positions.get('zero_deg')
        ninety_quat = positions.get('ninety_deg')
        bottom_side_pos = positions.get('bottom_side_pos')
        bottom_anchor_pos = positions.get('bottom_anchor_pos')
        tomato1_pos = positions.get('tomato1')
        tomato2_pos = positions.get('tomato2')
        plate_pos = positions.get('plate')
        
        if zero_quat is None or ninety_quat is None:
            raise ValueError("Missing orientation quaternions in positions dict")
        if bottom_side_pos is None or bottom_anchor_pos is None:
            raise ValueError("Missing drawer handle positions in positions dict")
        if tomato1_pos is None or tomato2_pos is None or plate_pos is None:
            raise ValueError("Missing object positions for tomatoes or plate")
        
        # === Execute Oracle Plan ===
        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Plan] Terminated after rotate. Reward:", reward)
            return

        # Step 2: move to side position of bottom drawer
        print("[Plan] Step 2: move to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Plan] Terminated after move-to-side. Reward:", reward)
            return

        # Step 3: move to anchor position of bottom drawer
        print("[Plan] Step 3: move to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Terminated after move-to-anchor. Reward:", reward)
            return

        # Step 4: pick the drawer handle (pick-drawer)
        print("[Plan] Step 4: pick drawer handle at:", bottom_anchor_pos)
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Terminated after pick-drawer. Reward:", reward)
            return

        # Step 5: pull the drawer open
        # We assume a small pull distance along the x-axis; adjust if needed
        pull_distance = 0.05
        print(f"[Plan] Step 5: pull drawer by {pull_distance} along 'x'")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Plan] Terminated after pull. Reward:", reward)
            return

        # Step 6: pick tomato1 from table
        print("[Plan] Step 6: pick tomato1 at:", tomato1_pos)
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Plan] Terminated after pick tomato1. Reward:", reward)
            return

        # Step 7: place tomato1 on plate
        print("[Plan] Step 7: place tomato1 at plate:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Completed after place tomato1. Reward:", reward)
            return

        # Step 8: pick tomato2 from table
        print("[Plan] Step 8: pick tomato2 at:", tomato2_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Plan] Terminated after pick tomato2. Reward:", reward)
            return

        # Step 9: place tomato2 on plate
        print("[Plan] Step 9: place tomato2 at plate:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Completed after place tomato2. Reward:", reward)
            return

        # Final check
        if not done:
            print("[Plan] All steps executed. Final reward:", reward)
        else:
            print("[Plan] Done flag raised at end. Reward:", reward)

    except Exception as e:
        print("Exception during plan execution:", str(e))
        raise
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
