[Plain text for run_skeleton_task.py]

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

# We need Rotation to build target quaternion for the rotate skill
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # === Oracle Plan Execution ===

        # Step 1: rotate gripper from zero to ninety degrees
        print("[Plan] Step 1: rotate gripper to 90 degrees around Z axis")
        # Construct a quaternion representing 90° rotation around Z
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Terminated early after rotate; reward:", reward)
            return

        # Step 2: move gripper to the side position of the bottom drawer
        print("[Plan] Step 2: move to bottom_side_pos")
        side_pos = positions['bottom_side_pos']
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Terminated early after move-to-side; reward:", reward)
            return

        # Step 3: move gripper to the anchor position of the bottom drawer
        print("[Plan] Step 3: move to bottom_anchor_pos")
        anchor_pos = positions['bottom_anchor_pos']
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Terminated early after move-to-anchor; reward:", reward)
            return

        # Step 4: pick the drawer handle at the anchor position
        print("[Plan] Step 4: pick-drawer (grasp handle)")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Plan] Terminated early after pick-drawer; reward:", reward)
            return

        # Step 5: pull to open the drawer
        print("[Plan] Step 5: pull to open drawer")
        # Pull along the +X axis by 0.1m (adjust as needed)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Terminated early after pull; reward:", reward)
            return

        # Step 6: pick tomato1 from the table
        print("[Plan] Step 6: pick tomato1")
        tomato1_pos = positions['tomato1']
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Plan] Terminated early after picking tomato1; reward:", reward)
            return

        # Step 7: place tomato1 onto the plate
        print("[Plan] Step 7: place tomato1 on plate")
        plate_pos = positions['plate']
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Terminated early after placing tomato1; reward:", reward)
            return

        # Step 8: pick tomato2 from the table
        print("[Plan] Step 8: pick tomato2")
        tomato2_pos = positions['tomato2']
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Plan] Terminated early after picking tomato2; reward:", reward)
            return

        # Step 9: place tomato2 onto the plate
        print("[Plan] Step 9: place tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Completed all steps with done=True. Final reward:", reward)
        else:
            print("[Task] Completed all steps but task not flagged done. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()