import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

[Frozen Code Start]
[Frozen Code End]

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper orientation to align with drawer handle
        print("[Plan] Step 1: rotate gripper to drawer alignment")
        current_obs = task.get_observation()
        current_quat = current_obs.gripper_pose[3:7]
        target_quat = current_quat  # no-op rotation to satisfy the step
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task ended after rotate. Exiting.")
            return

        # Step 2: Move to drawer side position
        print("[Plan] Step 2: move to drawer side position")
        side_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Task ended after move to side. Exiting.")
            return

        # Step 3: Move to drawer anchor position (for engaging the handle)
        print("[Plan] Step 3: move to drawer anchor position")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended after move to anchor. Exiting.")
            return

        # Step 4: Pull drawer open
        print("[Plan] Step 4: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Task ended after pull. Exiting.")
            return

        # Step 5: Pick up first tomato (item1)
        print("[Plan] Step 5: pick up item1")
        pos_item1 = np.array(positions['item1'])
        obs, reward, done = pick(env, task, pos_item1)
        if done:
            print("[Plan] Task ended during pick of item1. Exiting.")
            return

        # Step 6: Place first tomato on plate
        print("[Plan] Step 6: place item1 on plate")
        plate_pos = np.array(positions['plate'])
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Task ended during place of item1. Exiting.")
            return

        # Step 7: Pick up second tomato (item2)
        print("[Plan] Step 7: pick up item2")
        pos_item2 = np.array(positions['item2'])
        obs, reward, done = pick(env, task, pos_item2)
        if done:
            print("[Plan] Task ended during pick of item2. Exiting.")
            return

        # Step 8: Place second tomato on plate
        print("[Plan] Step 8: place item2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Task ended during place of item2. Exiting.")
            return

        print("[Plan] Completed all plan steps. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()