import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task and prepare video recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions
        positions = get_object_positions()
        bottom_side_pos   = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        bottom_joint_pos  = np.array(positions['bottom_joint_pos'])
        item1_pos         = np.array(positions['item1'])
        item2_pos         = np.array(positions['item2'])
        plate_pos         = np.array(positions['plate'])

        # === Plan Execution ===

        # Step 1: rotate gripper to its current orientation (no-op)
        print("[Plan] Step 1: rotate gripper to current orientation")
        current_quat = obs.gripper_pose[3:7]
        obs, reward, done = rotate(env, task, target_quat=current_quat)
        if done:
            print("[Plan] Task ended prematurely after rotate.")
            return

        # Step 2: move to drawer side position
        print(f"[Plan] Step 2: move to drawer side position at {bottom_side_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Plan] Task ended prematurely after move-to-side.")
            return

        # Step 3: move to drawer anchor position
        print(f"[Plan] Step 3: move to drawer anchor position at {bottom_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Task ended prematurely after move-to-anchor.")
            return

        # Step 4: pick the drawer handle
        print(f"[Plan] Step 4: pick drawer handle at {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Task ended prematurely after pick-drawer.")
            return

        # Step 5: pull the drawer open
        pull_distance = np.linalg.norm(bottom_joint_pos - bottom_anchor_pos)
        print(f"[Plan] Step 5: pull drawer open by {pull_distance:.3f}m")
        obs, reward, done = pull(env, task, pull_distance=pull_distance)
        if done:
            print("[Plan] Task ended prematurely after pull.")
            return

        # Step 6: pick tomato1
        print(f"[Plan] Step 6: pick tomato1 at {item1_pos}")
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        if done:
            print("[Plan] Task ended prematurely after pick tomato1.")
            return

        # Step 7: place tomato1 on plate
        print(f"[Plan] Step 7: place tomato1 on plate at {plate_pos}")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Task ended prematurely after place tomato1.")
            return

        # Step 8: pick tomato2
        print(f"[Plan] Step 8: pick tomato2 at {item2_pos}")
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        if done:
            print("[Plan] Task ended prematurely after pick tomato2.")
            return

        # Step 9: place tomato2 on plate
        print(f"[Plan] Step 9: place tomato2 on plate at {plate_pos}")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Task ended prematurely after place tomato2.")
            return

        print("[Plan] All steps executed. Task should be complete.")

    except Exception as e:
        print(f"[Error] Exception during execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()