from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Oracle Plan Task =====")
    env, task = setup_environment()
    try:
        # Reset and initialize recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve positions
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos       = positions['item1']
        tomato2_pos       = positions['item2']
        plate_pos         = positions['plate']

        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: rotate gripper to 90 degrees")
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()
[Frozen Code Start]
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        obs, reward, done = move(env, task, bottom_side_pos)
[Frozen Code End]
        if done:
            print("[Task] Ended after initial rotate/move.")
            return

        # Step 3: move to bottom drawer anchor position
        print("[Task] Step 3: move to bottom drawer anchor position")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended after move-to-anchor.")
            return

        # Step 4: grasp the drawer handle (pick-drawer)
        print("[Task] Step 4: pick-drawer (grasp drawer handle)")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended after pick-drawer.")
            return

        # Step 5: pull drawer open along the x-axis by 0.1m
        print("[Task] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ended after pull.")
            return

        # Step 6: pick the first tomato from the table
        print("[Task] Step 6: pick tomato1 from table")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Ended after picking tomato1.")
            return

        # Step 7: place the first tomato onto the plate
        print("[Task] Step 7: place tomato1 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended after placing tomato1.")
            return

        # Step 8: pick the second tomato from the table
        print("[Task] Step 8: pick tomato2 from table")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Ended after picking tomato2.")
            return

        # Step 9: place the second tomato onto the plate
        print("[Task] Step 9: place tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended after placing tomato2.")
            return

        print("[Task] Completed successfully! All tomatoes are on the plate.")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Oracle Plan Task =====")

if __name__ == "__main__":
    run_skeleton_task()