import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task and initialize video recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve positions of all relevant objects
        positions = get_object_positions()
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # === Execute Oracle Plan ===

        # Step 1: rotate gripper from zero_deg to ninety_deg
        quat_ninety = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        quat_ninety = normalize_quaternion(quat_ninety)
        print("[Task] Step 1: rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, quat_ninety)
        if done:
            print("[Task] Ended early at Step 1. Reward:", reward)
            return

        # Step 2: move-to-side (approach the drawer side)
        print("[Task] Step 2: move to bottom side position")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Ended early at Step 2. Reward:", reward)
            return

        # Step 3: move-to-anchor (approach the drawer handle)
        print("[Task] Step 3: move to bottom anchor position")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended early at Step 3. Reward:", reward)
            return

        # Step 4: pick-drawer (grasp the drawer handle)
        print("[Task] Step 4: pick the drawer handle")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended early at Step 4. Reward:", reward)
            return

        # Step 5: pull (open the drawer)
        print("[Task] Step 5: pull to open the drawer")
        # pull_distance and axis chosen to match drawer opening direction
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Task] Ended early at Step 5. Reward:", reward)
            return

        # Step 6: pick tomato1 from the table
        print("[Task] Step 6: pick tomato1")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Ended early at Step 6. Reward:", reward)
            return

        # Step 7: place tomato1 onto the plate
        print("[Task] Step 7: place tomato1 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended early at Step 7. Reward:", reward)
            return

        # Step 8: pick tomato2 from the table
        print("[Task] Step 8: pick tomato2")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Ended early at Step 8. Reward:", reward)
            return

        # Step 9: place tomato2 onto the plate
        print("[Task] Step 9: place tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Plan executed but not done=True. Reward:", reward)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()