import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the oracle plan to open a drawer and move tomatoes.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos       = positions['tomato1']
        tomato2_pos       = positions['tomato2']
        plate_pos         = positions['plate']

        # Step 1: rotate gripper 0° → 90° about Z
        print("[Step 1] Rotating gripper to 90 degrees about Z")
        zero_quat   = R.from_euler('z', 0,  degrees=True).as_quat()
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()


        obs, reward, done = rotate(env, task, ninety_quat)
        obs, reward, done = move(env, task, bottom_side_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Step 3] Task ended prematurely.")
            return

        # Step 4: pick the drawer handle
        print("[Step 4] Grasping the drawer handle")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Step 4] Task ended prematurely.")
            return

        # Step 5: pull drawer open by 0.1m along X
        print("[Step 5] Pulling the drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Step 5] Task ended prematurely.")
            return

        # Step 6: pick tomato1
        print(f"[Step 6] Picking tomato1 at {tomato1_pos}")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Step 6] Task ended prematurely.")
            return

        # Step 7: place tomato1 on plate
        print(f"[Step 7] Placing tomato1 at {plate_pos}")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Step 7] Task ended prematurely.")
            return

        # Step 8: pick tomato2
        print(f"[Step 8] Picking tomato2 at {tomato2_pos}")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Step 8] Task ended prematurely.")
            return

        # Step 9: place tomato2 on plate
        print(f"[Step 9] Placing tomato2 at {plate_pos}")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Step 9] Task ended prematurely.")
            return

        print("===== Oracle plan executed successfully! Final reward:", reward)

    except Exception as e:
        print("Exception during task execution:", str(e))
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()