import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve all object positions
        positions = get_object_positions()

        # Prepare key positions and orientations
        curr_quat = obs.gripper_pose[3:7]
        side_pos = positions['bottom_side_pos']
        pull_distance = 0.1
        pull_axis = 'x'

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper (dummy rotate to current orientation)
        print("[Task] Step 1: rotate gripper to current orientation")
        obs, reward, done = rotate(env, task, curr_quat)
        if done:
            print("[Task] Task ended after rotate.")
            return

        # Step 2: Move to the side position of the bottom drawer
        print(f"[Task] Step 2: move to bottom side position {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended after move to side.")
            return

        # Step 3: Pull to open the drawer
        print("[Task] Step 3: pull to open drawer")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Task] Task ended after pull.")
            return

        # Step 4: Pick tomato1 from the table
        tomato1_pos = positions['tomato1']
        print(f"[Task] Step 4: pick tomato1 at {tomato1_pos}")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Task ended after picking tomato1.")
            return

        # Step 5: Place tomato1 on the plate
        plate_pos = positions['plate']
        print(f"[Task] Step 5: place tomato1 on plate at {plate_pos}")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Task ended after placing tomato1.")
            return

        # Step 6: Pick tomato2 from the table
        tomato2_pos = positions['tomato2']
        print(f"[Task] Step 6: pick tomato2 at {tomato2_pos}")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Task ended after picking tomato2.")
            return

        # Step 7: Place tomato2 on the plate
        print(f"[Task] Step 7: place tomato2 on plate at {plate_pos}")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Task ended after placing tomato2.")
            return

        # Final check
        if reward is not None:
            print(f"[Task] Completed plan. Final reward: {reward}")
        else:
            print("[Task] Completed plan.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()