import numpy as np  
from pyrep.objects.shape import Shape  
from pyrep.objects.proximity_sensor import ProximitySensor  

from env import setup_environment, shutdown_environment  

from skill_code import *  

from video import init_video_writers, recording_step, recording_get_observation  

from object_positions import get_object_positions  

def run_skeleton_task():  
    '''Oracle drawer-and-toss task: open a drawer, then toss two tomatoes onto a plate.'''  
    print("===== Starting Skeleton Task =====")  

    # === Environment Setup ===  
    env, task = setup_environment()  
    try:  
        # Reset the task  
        descriptions, obs = task.reset()  

        # Initialize video recording  
        init_video_writers(obs)  

        # Wrap step & observation for recording  
        task.step = recording_step(task.step)  
        task.get_observation = recording_get_observation(task.get_observation)  

        # === Retrieve Object Positions ===  
        positions = get_object_positions()  
        side_bottom_pos = positions['bottom_side_pos']  
        anchor_bottom_pos = positions['bottom_anchor_pos']  
        tomato1_pos = positions['tomato1']  
        tomato2_pos = positions['tomato2']  
        plate_pos = positions['plate']  

        # === Prepare orientation quaternions ===  
        zero_quat = np.array(Shape('zero_deg').get_quaternion())  
        ninety_quat = np.array(Shape('ninety_deg').get_quaternion())  

        # === Oracle Plan Execution ===  

        # Step 1: rotate from zero_deg to ninety_deg  
        print("[Step 1] rotate gripper to ninety_deg")  
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)  
        if done: return  

        # Step 2: move to side position of bottom drawer  
        print("[Step 2] move to side-bottom position:", side_bottom_pos)  
        obs, reward, done = move(env, task, target_pos=np.array(side_bottom_pos))  
        if done: return  

        # Step 3: move to anchor position of bottom drawer  
        print("[Step 3] move to anchor-bottom position:", anchor_bottom_pos)  
        obs, reward, done = move(env, task, target_pos=np.array(anchor_bottom_pos))  
        if done: return  

        # Step 4: pick the drawer handle at anchor position  
        print("[Step 4] pick drawer handle at anchor-bottom position")  
        obs, reward, done = pick(env, task, target_pos=np.array(anchor_bottom_pos))  
        if done: return  

        # Step 5: pull the drawer open along +x axis  
        print("[Step 5] pull drawer open")  
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')  
        if done: return  

        # Step 6: pick tomato1 from table  
        print("[Step 6] pick tomato1 at:", tomato1_pos)  
        obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos))  
        if done: return  

        # Step 7: place tomato1 onto the plate  
        print("[Step 7] place tomato1 on plate at:", plate_pos)  
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))  
        if done: return  

        # Step 8: pick tomato2 from table  
        print("[Step 8] pick tomato2 at:", tomato2_pos)  
        obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos))  
        if done: return  

        # Step 9: place tomato2 on plate  
        print("[Step 9] place tomato2 on plate at:", plate_pos)  
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))  

        print("===== Plan execution complete! =====")  
        print("Final reward:", reward, "Done flag:", done)  

    except Exception as e:  
        print("Exception during task execution:", str(e))  
        raise  
    finally:  
        shutdown_environment(env)  
        print("===== End of Skeleton Task =====")  

if __name__ == "__main__":  
    run_skeleton_task()