import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull, normalize_quaternion
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_pos_bottom   = np.array(positions['bottom_side_pos'])
        anchor_pos_bottom = np.array(positions['bottom_anchor_pos'])
        plate_pos         = np.array(positions['plate'])
        tomato_names      = ['tomato1', 'tomato2']

        # === Oracle Plan Execution ===

        # Step 1: Move to side position of bottom drawer
        print("[Plan] Step 1: move to bottom side position")
[Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=side_pos_bottom)
        obs, reward, done = rotate(env, task, target_quat=target_quat)
[Frozen Code End]
        if done:
            print("[Plan] Early termination after step 1 or 2.")
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Plan] Step 3: move to bottom anchor position")
        obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)
        if done:
            print("[Plan] Early termination after move to anchor.")
            return

        # Step 4: Pick the drawer handle at anchor
        print("[Plan] Step 4: pick drawer handle at anchor")
        obs, reward, done = pick(env, task, target_pos=anchor_pos_bottom)
        if done:
            print("[Plan] Early termination after picking drawer handle.")
            return

        # Step 5: Pull the bottom drawer open
        print("[Plan] Step 5: pull bottom drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1)
        if done:
            print("[Plan] Early termination after pull.")
            return

        # Steps 6–9: Pick each tomato and place it on the plate
        step_counter = 6
        for name in tomato_names:
            obj_pos = np.array(positions[name])
            print(f"[Plan] Step {step_counter}: pick {name}")
            obs, reward, done = pick(env, task, target_pos=obj_pos)
            if done:
                print(f"[Plan] Early termination after picking {name}.")
                return
            step_counter += 1

            print(f"[Plan] Step {step_counter}: place {name} on plate")
            obs, reward, done = place(env, task, target_pos=plate_pos)
            if done:
                print(f"[Plan] Early termination after placing {name}.")
                return
            step_counter += 1

        print("[Plan] All steps executed. Task should be complete.")

    finally:
        shutdown_environment(env)

if __name__ == "__main__":
    run_skeleton_task()