import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expected keys:
        # 'bottom_side_pos', 'bottom_anchor_pos', 'tomato1', 'tomato2', 'plate'
        side_pos = positions['bottom_side_pos']
        anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # === Prepare orientations ===
        zero_shape = Shape('zero_deg')
        ninety_shape = Shape('ninety_deg')
        ninety_quat = ninety_shape.get_quaternion()  # [x, y, z, w]

        # === Oracle Plan Execution ===

        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        obs, reward, done = move(env, task, target_pos=side_pos)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        # [Frozen Code End]

        # Step 9: Place tomato2 on plate
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Done after placing tomato2.")
            return

        print("[Task] Completed all steps successfully. Final reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
        raise

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
