import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task and initialize video recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()
        # Bottom drawer side and anchor positions
        bottom_side = positions.get('bottom_side_pos')
        bottom_anchor = positions.get('bottom_anchor_pos')
        if bottom_side is None or bottom_anchor is None:
            raise KeyError("Could not find bottom drawer positions in get_object_positions()")

        # Tomatoes and plate
        tomato1_pos = positions.get('tomato1')
        tomato2_pos = positions.get('tomato2')
        plate_pos = positions.get('plate')
        if tomato1_pos is None or tomato2_pos is None or plate_pos is None:
            raise KeyError("Could not find tomato or plate positions in get_object_positions()")

        # Define quaternions for 0° and 90° around Z axis
        zero_quat = R.from_euler('xyz', [0, 0, 0], degrees=True).as_quat()
        ninety_quat = R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()

        # Step 1: Rotate gripper to 90°
        print("[Step 1] Rotating gripper to 90° about Z axis")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("Task ended prematurely after rotation.")
            return

        # Step 2: Move to side position of bottom drawer
        print("[Step 2] Moving gripper to bottom drawer side position")
        obs, reward, done = move(env, task, bottom_side)
        if done:
            print("Task ended prematurely after move to side position.")
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Step 3] Moving gripper to bottom drawer anchor position")
        obs, reward, done = move(env, task, bottom_anchor)
        if done:
            print("Task ended prematurely after move to anchor position.")
            return

        # Step 4: Pick the drawer handle at the anchor position
        print("[Step 4] Picking drawer handle at anchor position")
        obs, reward, done = pick(env, task, bottom_anchor)
        if done:
            print("Task ended prematurely after picking drawer handle.")
            return

        # Step 5: Pull the drawer open
        print("[Step 5] Pulling drawer open")
        # Adjust distance and axis to match drawer geometry
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("Task ended prematurely after pulling drawer.")
            return

        # Step 6: Pick tomato1 from table
        print("[Step 6] Picking tomato1 from table")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("Task ended prematurely after picking tomato1.")
            return

        # Step 7: Place tomato1 on plate
        print("[Step 7] Placing tomato1 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("Task ended prematurely after placing tomato1.")
            return

        # Step 8: Pick tomato2 from table
        print("[Step 8] Picking tomato2 from table")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("Task ended prematurely after picking tomato2.")
            return

        # Step 9: Place tomato2 on plate
        print("[Step 9] Placing tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("Task ended prematurely after placing tomato2.")
            return

        print("Task completed successfully! Final reward:", reward)

    except Exception as e:
        print("Exception during execution:", e)
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()