import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *  # predefined skills: move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve all object positions
        positions = get_object_positions()
        # Assume these keys exist in positions dict
        #   'gripper'      : current gripper position/orientation
        #   'drawer'       : the drawer object
        #   'side_pos'     : approach position at side of drawer
        #   'anchor_pos'   : precise grasp position on drawer handle
        gripper_name = 'gripper'
        drawer_name = 'drawer'
        side_pos = positions['side_pos']
        anchor_pos = positions['anchor_pos']

        # 1) Rotate gripper so it is aligned for side approach
        #    here we assume ninety_deg quaternion is known or computed
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to ninety_deg orientation")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Terminated during rotate.")
            return

        # 2) Move gripper to the side position of the drawer
        print(f"[Task] Moving to side position: {side_pos}")
        obs, reward, done = move(env, task, target_pos=side_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Terminated during move to side.")
            return

        # 3) Move gripper from side position to anchor position (handle)
        print(f"[Task] Moving to anchor position: {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=anchor_pos, max_steps=100, threshold=0.005, timeout=10.0)
        if done:
            print("[Task] Terminated during move to anchor.")
            return

        # 4) Grasp the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.02, max_steps=100, threshold=0.01, timeout=5.0)
        if done:
            print("[Task] Terminated during pick.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer")
        # we expect pull to open the drawer when holding the handle
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Terminated during pull.")
            return

        # 6) Exploration phase: check if we know the lock state of the drawer
        #    if we failed to pull because lock is unknown or locked, perform sensing
        try:
            # Attempt to re-pull to see if the drawer is already open
            obs, reward, done = pull(env, task)
            print("[Exploration] pull succeeded, drawer open.")
        except Exception as e:
            print(f"[Exploration] pull failed with error: {e}")
            print("[Exploration] Performing extra sensing on drawer lock state.")
            # e.g., pick then place handle for weight/durability/lock feedback
            obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.02, max_steps=50, threshold=0.01, timeout=5.0)
            obs, reward, done = place(env, task, target_pos=anchor_pos, max_steps=50, threshold=0.01, timeout=5.0)
            print("[Exploration] Retry pulling after sensing.")
            obs, reward, done = pull(env, task)
            if done:
                print("[Task] Drawer pulled open after exploration.")

        print("===== Task Completed: Drawer is open =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()