import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # fetch all known positions/quaternions for objects
        positions = get_object_positions()
        print("[Setup] Retrieved positions:", positions.keys())

        # Identify the drawer object name in the positions dict
        drawer_key = None
        for key in positions:
            if 'drawer' in key and 'pos' not in key and 'quat' not in key:
                drawer_key = key
                break
        if drawer_key is None:
            # fallback to a generic name
            drawer_key = 'drawer'
        print(f"[Setup] Using drawer identifier: {drawer_key}")

        # Find side positions for the drawer
        side_keys = [k for k in positions if 'side' in k and drawer_key in k]
        if len(side_keys) < 2:
            raise RuntimeError("Expected at least two side positions for move-to-side but found: " + str(side_keys))
        # assume ordering: first is starting side, second is target side
        side_from_key, side_to_key = side_keys[0], side_keys[1]
        side_from = positions[side_from_key]
        side_to = positions[side_to_key]
        print(f"[Setup] side_from: {side_from_key}, side_to: {side_to_key}")

        # Find anchor positions for the drawer
        anchor_keys = [k for k in positions if 'anchor' in k and drawer_key in k]
        if len(anchor_keys) < 2:
            raise RuntimeError("Expected at least two anchor positions for move-to-anchor but found: " + str(anchor_keys))
        anchor_from_key, anchor_to_key = anchor_keys[0], anchor_keys[1]
        anchor_from = positions[anchor_from_key]
        anchor_to = positions[anchor_to_key]
        print(f"[Setup] anchor_from: {anchor_from_key}, anchor_to: {anchor_to_key}")

        # Find the target quaternion for rotating the gripper
        quat_key = None
        for k in positions:
            if 'quat' in k or 'rotation' in k or 'ninety' in k:
                quat_key = k
                break
        if quat_key is None:
            raise RuntimeError("No target quaternion found in positions for rotation.")
        target_quat = positions[quat_key]
        print(f"[Setup] Using target orientation quat from key: {quat_key}")

        # STEP 1: Rotate the gripper to the desired orientation
        print("[Plan] Rotating gripper to target orientation.")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task terminated during rotate.")
            return

        # STEP 2: Move from current side position to the side of the drawer
        print(f"[Plan] Moving gripper to side position of drawer: from {side_from_key} to {side_to_key}")
        obs, reward, done = move_to_side(env, task, drawer_key, side_from, side_to)
        if done:
            print("[Plan] Task terminated during move_to_side.")
            return

        # STEP 3: Move from side position to anchor position on the drawer handle
        print(f"[Plan] Moving gripper from side to anchor pos: {anchor_to_key}")
        obs, reward, done = move_to_anchor(env, task, drawer_key, anchor_from, anchor_to)
        if done:
            print("[Plan] Task terminated during move_to_anchor.")
            return

        # STEP 4: Pick the drawer handle at the anchor position
        print(f"[Plan] Grasping drawer handle at: {anchor_to_key}")
        obs, reward, done = pick_drawer(env, task, drawer_key, anchor_to)
        if done:
            print("[Plan] Task terminated during pick_drawer.")
            return

        # STEP 5: Pull the drawer open
        print("[Plan] Pulling the drawer open.")
        obs, reward, done = pull(env, task, drawer_key)
        if done:
            print("[Plan] Task terminated during pull.")
            return

        print("[Plan] Drawer should now be open.")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()