import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()
        # Check that both tomatoes and plate are present
        tomato_names = [name for name in positions if 'tomato' in name]
        if len(tomato_names) < 2:
            print("[Error] Less than two tomatoes found in positions:", positions.keys())
            return
        plate_name = next((n for n in positions if 'plate' in n), None)
        if plate_name is None:
            print("[Error] Plate not found in environment.")
            return

        tomato1_pos = positions[tomato_names[0]]
        tomato2_pos = positions[tomato_names[1]]
        plate_pos = positions[plate_name]

        # If plate is inside a drawer, open the drawer first
        if 'drawer' in positions:
            drawer_anchor = positions.get('drawer_anchor_pos')
            drawer_side = positions.get('drawer_side_pos')
            # 1) Rotate gripper to ninety degrees to align with handle
            print("[Task] Rotating gripper to open drawer handle")
            obs, reward, done = rotate(env, task,
                                       target_quat=[0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)],
                                       max_steps=100, threshold=0.05, timeout=10.0)
            if done:
                print("[Task] Early termination after rotate.")
                return
            # 2) Move gripper to side position of drawer
            if drawer_side is not None:
                print(f"[Task] Moving gripper to drawer side at {drawer_side}")
                obs, reward, done = move(env, task,
                                         target_pos=drawer_side,
                                         approach_distance=0.05,
                                         max_steps=100, threshold=0.01,
                                         approach_axis='z', timeout=10.0)
                if done:
                    print("[Task] Early termination after move-to-side.")
                    return
            # 3) Move gripper to anchor position
            if drawer_anchor is not None:
                print(f"[Task] Moving gripper to drawer anchor at {drawer_anchor}")
                obs, reward, done = move(env, task,
                                         target_pos=drawer_anchor,
                                         approach_distance=0.02,
                                         max_steps=100, threshold=0.005,
                                         approach_axis='z', timeout=10.0)
                if done:
                    print("[Task] Early termination after move-to-anchor.")
                    return
                # 4) Pick the drawer handle
                obs, reward, done = pick(env, task,
                                         target_pos=drawer_anchor,
                                         approach_distance=0.02,
                                         max_steps=100, threshold=0.005,
                                         approach_axis='z', timeout=5.0)
                if done:
                    print("[Task] Early termination after pick-drawer.")
                    return
                # 5) Pull the drawer open
                print("[Task] Pulling drawer open")
                obs, reward, done = pull(env, task)
                if done:
                    print("[Task] Early termination after pulling drawer.")
                    return

        # Now pick and place each tomato onto the plate
        for tname, tpos in zip(tomato_names, [tomato1_pos, tomato2_pos]):
            print(f"[Task] Picking {tname} at {tpos}")
            obs, reward, done = pick(env, task,
                                     target_pos=tpos,
                                     approach_distance=0.10,
                                     max_steps=100,
                                     threshold=0.01,
                                     approach_axis='z',
                                     timeout=10.0)
            if done:
                print(f"[Task] Early termination after picking {tname}.")
                return

            print(f"[Task] Placing {tname} on plate at {plate_pos}")
            obs, reward, done = place(env, task,
                                      target_pos=plate_pos,
                                      approach_distance=0.05,
                                      max_steps=100,
                                      threshold=0.01,
                                      approach_axis='z',
                                      timeout=10.0)
            if done:
                print(f"[Task] Early termination after placing {tname}.")
                return

        # Verification: check both tomatoes are on the plate
        obs = task.get_observation()
        # assume obs.contains returns list of names at a position
        on_plate = [obj for obj in getattr(obs, 'objects_on_plate', []) if 'tomato' in obj]
        if len(on_plate) == 2:
            print("[Task] Success: Both tomatoes are on the plate.")
        else:
            print(f"[Task] Failure: Expected 2 tomatoes on plate, found {len(on_plate)}: {on_plate}")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()