# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset task to initial state and set up recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # NOTE: Update these keys to match your environment’s actual object names
        gripper = 'gripper1'
        drawer = 'drawer1'
        initial_angle = positions['initial_angle']          # e.g. 'zero_deg'
        ninety_angle  = positions['ninety_deg']             # e.g. 'ninety_deg'
        side_position = positions['drawer_side_pos']        # a Position object or coordinate
        anchor_position = positions['drawer_anchor_pos']    # a Position object or coordinate
        # If you know the current side position of the gripper you can override:
        current_side_position = positions.get('current_side_pos', side_position)

        # === Oracle Plan Execution ===
        # 1) Rotate gripper from initial angle to 90°
        print("[Task] Rotating gripper from", initial_angle, "to", ninety_angle)
        obs, reward, done = rotate(env, task, gripper, initial_angle, ninety_angle)
        if done:
            print("[Task] Finished after rotate")
            return

        # 2) Move to drawer side position
        print("[Task] Moving gripper to side position:", side_position)
        obs, reward, done = move_to_side(env, task, gripper, drawer, current_side_position, side_position)
        if done:
            print("[Task] Finished after move_to_side")
            return

        # 3) Move to drawer anchor position
        print("[Task] Moving gripper to anchor position:", anchor_position)
        obs, reward, done = move_to_anchor(env, task, gripper, drawer, side_position, anchor_position)
        if done:
            print("[Task] Finished after move_to_anchor")
            return

        # 4) Grasp the drawer handle
        print("[Task] Picking up drawer handle")
        obs, reward, done = pick_drawer(env, task, gripper, drawer, anchor_position)
        if done:
            print("[Task] Finished after pick_drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task, gripper, drawer)
        if done:
            print("[Task] Finished after pull")
            return

        print("[Task] Drawer should now be open!")
    finally:
        # Always shut down environment
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()