import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace these keys with the actual ones returned by get_object_positions()
        object_1_pos = positions['object_1']
        object_2_pos = positions['object_2']
        drop_off_pos = positions['drop_off']
        drawer_side_pos = positions['drawer_side_position']
        drawer_anchor_pos = positions['drawer_anchor_position']

        # === Phase 1: Manipulate object_1 ===
        print("[Task] Approaching object_1 at:", object_1_pos)
        obs, reward, done = pick(
            env, task,
            target_pos=object_1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended after pick(object_1).")
            return

        print("[Task] Placing object_1 at drop_off:", drop_off_pos)
        obs, reward, done = place(env, task, target_pos=drop_off_pos)
        if done:
            print("[Task] Ended after place(object_1).")
            return

        # === Phase 2: Manipulate object_2 ===
        print("[Task] Approaching object_2 at:", object_2_pos)
        obs, reward, done = pick(
            env, task,
            target_pos=object_2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended after pick(object_2).")
            return

        print("[Task] Placing object_2 at drop_off:", drop_off_pos)
        obs, reward, done = place(env, task, target_pos=drop_off_pos)
        if done:
            print("[Task] Ended after place(object_2).")
            return

        # === Phase 3: Open Drawer ===
        gripper_name = 'gripper'
        zero_angle = 'zero_deg'
        ninety_angle = 'ninety_deg'

        print("[Task] Rotating gripper from", zero_angle, "to", ninety_angle)
        obs, reward, done = rotate(env, task, gripper_name, zero_angle, ninety_angle)
        if done:
            print("[Task] Ended after rotate.")
            return

        print("[Task] Moving gripper to side position:", drawer_side_pos)
        obs, reward, done = move(env, task, gripper_name, drawer_side_pos)
        if done:
            print("[Task] Ended after move-to-side.")
            return

        print("[Task] Moving gripper to anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, drawer_side_pos, drawer_anchor_pos)
        if done:
            print("[Task] Ended after move-to-anchor.")
            return

        print("[Task] Grasping drawer handle at:", drawer_anchor_pos)
        obs, reward, done = pick(
            env, task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.005,
            approach_axis='y',
            timeout=5.0
        )
        if done:
            print("[Task] Ended after pick-drawer.")
            return

        print("[Task] Pulling drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Ended after pull.")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()