import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running a drawer‐opening task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # reset to initial state
        descriptions, obs = task.reset()

        # initialize video capture
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve approximate object poses from helper module
        positions = get_object_positions()
        # these keys must match those returned by get_object_positions()
        # e.g. 'drawer_side', 'drawer_anchor', maybe 'gripper_quat_ninety'
        drawer_side_pos = positions['drawer_side']
        drawer_anchor_pos = positions['drawer_anchor']
        # target orientation quaternion for 90° rotation
        ninety_quat = positions.get('gripper_quat_ninety', [0.0, 0.0, 0.7071, 0.7071])

        # === Oracle Plan Execution ===

        # 1) Rotate gripper to face the side of the drawer
        print("[Plan] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Terminated during rotate")
            return

        # 2) Move gripper to the side position of the drawer
        print("[Plan] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Terminated during move to side")
            return

        # 3) Move gripper from side to anchor position
        print("[Plan] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Plan] Terminated during move to anchor")
            return

        # 4) Pick the drawer handle at anchor position
        print("[Plan] Picking drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.02,
            max_steps=50,
            threshold=0.005,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Plan] Terminated during pick")
            return

        # 5) Pull to open the drawer
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] Terminated during pull")
            return

        print("[Plan] Drawer should now be open. Task complete.")
        
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
