import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset and get domain descriptions
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # fetch all known object positions from the scene
        positions = get_object_positions()
        # we expect keys 'drawer_side' and 'drawer_anchor' here
        if 'drawer_side' not in positions or 'drawer_anchor' not in positions:
            raise KeyError("Expected 'drawer_side' or 'drawer_anchor' in positions but got: " + str(list(positions.keys())))
        side_pos = np.array(positions['drawer_side'])
        anchor_pos = np.array(positions['drawer_anchor'])

        # exploration phase: check if the predicate 'rotated' appears in the PDDL description
        desc_text = " ".join(descriptions)
        if 'rotated' not in desc_text:
            print("[Exploration] Predicate 'rotated' is missing in domain. Rotating gripper as exploration.")
            # assume gripper name is 'gripper0' and angles 'zero_deg' -> 'ninety_deg'
            obs, reward, done = rotate(env, task, 'gripper0', 'zero_deg', 'ninety_deg')
            if done:
                print("[Exploration] Task ended during rotation.")
                return
            # rotate it back so we don't disturb the later plan
            obs, reward, done = rotate(env, task, 'gripper0', 'ninety_deg', 'zero_deg')
            if done:
                print("[Exploration] Task ended during rotation back.")
                return
        else:
            print("[Exploration] Predicate 'rotated' found in domain.")

        # ===== Oracle Plan Execution =====
        # 1) set the gripper orientation to ninety degrees
        print("[Plan] rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, 'gripper0', 'zero_deg', 'ninety_deg')
        if done:
            print("[Plan] ended after rotate.")
            return

        # 2) move to the side position of the drawer
        print(f"[Plan] move to drawer side position {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] ended after moving to side.")
            return

        # 3) move to the anchor position of the drawer handle
        print(f"[Plan] move to drawer anchor position {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] ended after moving to anchor.")
            return

        # 4) pick the drawer handle
        print("[Plan] pick drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] ended after pick-drawer.")
            return

        # 5) pull the drawer open
        print("[Plan] pull drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] ended after pull.")
            return

        print("[Plan] Drawer should now be open. Task completed.")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()