import numpy as np
import math
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step & get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect keys for drawer, side position, anchor position, maybe handle
        drawer_base_pos = positions.get('drawer', None)
        drawer_side_pos = positions.get('drawer_side_pos', None)
        drawer_anchor_pos = positions.get('drawer_anchor_pos', None)
        drawer_handle_pos = positions.get('drawer_handle', drawer_anchor_pos)

        if drawer_side_pos is None or drawer_anchor_pos is None:
            raise ValueError("Missing drawer side/anchor positions in object_positions")

        # === Oracle Plan Execution ===
        # 1) Exploration: ensure gripper rotated zero_deg to identify any missing predicate
        print("[Plan] Rotate gripper to zero_deg for exploration")
        zero_quat = [0.0, 0.0, 0.0, 1.0]
        obs, reward, done = rotate(env, task, zero_quat)
        if done:
            print("[Plan] Task ended prematurely after zero_deg rotate")
            return

        # 2) Move to side of drawer
        print(f"[Plan] Moving to side position: {drawer_side_pos}")
        obs, reward, done = move(env, task,
                                  target_pos=drawer_side_pos,
                                  max_steps=150,
                                  threshold=0.01,
                                  timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely after move-to-side")
            return

        # 3) Rotate gripper to ninety_deg to align for picking the drawer handle
        print("[Plan] Rotate gripper to ninety_deg to align handle")
        # construct a quaternion for 90° around Z axis
        half_angle = math.pi / 4.0
        ninety_quat = [0.0, 0.0, math.sin(half_angle), math.cos(half_angle)]
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Task ended prematurely after ninety_deg rotate")
            return

        # 4) Move from side to anchor position on the drawer
        print(f"[Plan] Moving to anchor position: {drawer_anchor_pos}")
        obs, reward, done = move(env, task,
                                  target_pos=drawer_anchor_pos,
                                  max_steps=150,
                                  threshold=0.01,
                                  timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely after move-to-anchor")
            return

        # 5) Pick the drawer handle
        print(f"[Plan] Picking drawer handle at: {drawer_handle_pos}")
        obs, reward, done = pick(env, task,
                                  target_pos=drawer_handle_pos,
                                  approach_distance=0.05,
                                  max_steps=100,
                                  threshold=0.01,
                                  approach_axis='z',
                                  timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely after pick handle")
            return

        # 6) Pull the drawer open
        print("[Plan] Pulling the drawer open")
        obs, reward, done = pull(env, task,
                                  max_steps=80,
                                  threshold=0.005,
                                  timeout=8.0)
        if done:
            print("[Plan] Task ended prematurely after pull")
            return

        # 7) (Optional) Place any object or finalize task
        print("[Plan] Task plan completed; drawer should be open.")

    except Exception as e:
        print(f"[Error] Exception during plan execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()