import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap task.step and task.get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve object positions from the scene
        positions = get_object_positions()
        # expect keys 'drawer_side_pos' and 'drawer_anchor_pos'
        side_pos = positions.get('drawer_side_pos')
        anchor_pos = positions.get('drawer_anchor_pos')
        if side_pos is None or anchor_pos is None:
            print("Error: drawer handle positions not found in object_positions")
            return

        # 1) rotate gripper to side orientation (90 degrees)
        print("[Task] Rotating gripper to side orientation")
        # quaternion for a 90-degree rotation around Z
        target_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        obs, reward, done = rotate(env, task, target_quat, max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Episode ended during rotate")
            return

        # 2) move to the side position of the drawer handle
        print("[Task] Moving to drawer side position:", side_pos)
        obs, reward, done = move(env, task,
                                 target_pos=side_pos,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Episode ended during move to side")
            return

        # 3) move to the anchor position of the drawer handle
        print("[Task] Moving to drawer anchor position:", anchor_pos)
        obs, reward, done = move(env, task,
                                 target_pos=anchor_pos,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Episode ended during move to anchor")
            return

        # 4) pick the drawer handle
        print("[Task] Picking the drawer handle at anchor position")
        obs, reward, done = pick(env, task,
                                 target_pos=anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=50,
                                 threshold=0.005,
                                 approach_axis='x',
                                 timeout=5.0)
        if done:
            print("[Task] Episode ended during pick")
            return

        # 5) pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task,
                                 max_steps=50,
                                 force=5.0,
                                 timeout=5.0)
        if done:
            print("[Task] Drawer pull action ended episode")
            return

        print("[Task] Completed plan: drawer should now be open")

    except Exception as e:
        print("Error during task execution:", e)
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()