import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve positions we need for the drawer task
        positions = get_object_positions()
        try:
            drawer_side_pos = positions['drawer_side_pos']
            drawer_anchor_pos = positions['drawer_anchor_pos']
            # if available, a canonical quaternion for 90° rotation
            drawer_ninety_quat = positions.get('drawer_ninety_quat', [0, 0, 0, 1])
        except KeyError as e:
            print(f"[Error] Missing required position in object_positions: {e}")
            return

        # 1) Rotate gripper into alignment with drawer side
        print("[Task] Rotating gripper to 90° orientation for drawer side approach.")
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=drawer_ninety_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            return
        if done:
            print("[Task] Episode ended during rotation.")
            return

        # 2) Move gripper to the side position of the drawer
        print("[Task] Moving gripper to side position of drawer.")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=drawer_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during move to side position: {e}")
            return
        if done:
            print("[Task] Episode ended during move-to-side.")
            return

        # 3) Move gripper to the anchor position (drawer handle)
        print("[Task] Moving gripper to anchor position (drawer handle).")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=drawer_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during move to anchor position: {e}")
            return
        if done:
            print("[Task] Episode ended during move-to-anchor.")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking up the drawer handle.")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=drawer_anchor_pos,
                approach_distance=0.02,
                max_steps=100,
                threshold=0.005,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during pick drawer handle: {e}")
            return
        if done:
            print("[Task] Episode ended during pick operation.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open.")
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            return
        if done:
            print("[Task] Drawer successfully opened; task complete!")
            return

        print("[Task] Finished all steps without episode termination.")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()