import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace these keys with the actual names used in positions
        home_pos = positions.get('home_pos')
        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']
        drawer_handle = positions.get('drawer_handle')  # if provided

        # === Define Oracle Plan ===
        # We assume the gripper starts at angle 'zero_deg' and needs to rotate to 'ninety_deg'
        start_angle = 'zero_deg'
        target_angle = 'ninety_deg'
        plan = [
            # 1) Rotate gripper to approach drawer from side
            ('rotate', dict(env=env, task=task, g='gripper', from_angle=start_angle, to_angle=target_angle)),
            # 2) Move gripper from home to drawer side
            ('move', dict(env=env, task=task, from_pos=home_pos, to_pos=side_pos)),
            # 3) Move gripper from side to anchor position on drawer
            ('move', dict(env=env, task=task, from_pos=side_pos, to_pos=anchor_pos)),
            # 4) Pick the drawer handle at the anchor position
            ('pick', dict(env=env,
                          task=task,
                          target_pos=anchor_pos,
                          approach_distance=0.15,
                          max_steps=100,
                          threshold=0.01,
                          approach_axis='z',
                          timeout=10.0)),
            # 5) Pull the drawer open
            ('pull', dict(env=env, task=task, g='gripper', d='drawer'))
        ]

        # === Execute the Plan Step by Step ===
        for step_idx, (action_name, params) in enumerate(plan):
            print(f"[Task] Step {step_idx+1}/{len(plan)}: Executing {action_name} with params {params}")
            if action_name == 'rotate':
                obs, reward, done = rotate(**params)
            elif action_name == 'move':
                obs, reward, done = move(**params)
            elif action_name == 'pick':
                obs, reward, done = pick(**params)
            elif action_name == 'place':
                obs, reward, done = place(**params)
            elif action_name == 'pull':
                obs, reward, done = pull(**params)
            else:
                raise ValueError(f"Unknown action '{action_name}' in plan")

            if done:
                print(f"[Task] Task ended prematurely after action '{action_name}'")
                return

        print("[Task] Plan executed successfully, drawer should now be open.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()