import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull, move_to_side, move_to_anchor, pick_drawer

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve positions for all relevant objects
        positions = get_object_positions()
        print("[Info] Available position keys:", list(positions.keys()))

        # define which gripper and drawer we're manipulating
        gripper = 'gripper1'
        drawer = 'bottom_drawer'

        # pull out the two essential poses for drawer opening
        side_pos = positions['bottom_side_pos']
        anchor_pos = positions['bottom_anchor_pos']

        # two discrete gripper angles (assuming these keys exist in positions dict)
        zero_deg = positions.get('zero_deg', None)
        ninety_deg = positions.get('ninety_deg', None)
        if zero_deg is None or ninety_deg is None:
            raise RuntimeError("Required gripper angles 'zero_deg' or 'ninety_deg' not found in positions")

        # === Exploration Phase: determine lock status ===
        print("[Exploration] Attempting to pull drawer without picking it to infer lock state")
        try:
            obs, reward, done = pull(env, task, gripper, drawer)
            if done:
                print("[Exploration] Task ended unexpectedly during exploration.")
                return
            print("[Exploration] Pulled directly -> drawer is not locked")
        except Exception as e:
            print(f"[Exploration] Pull failed ({e}) -> drawer may be locked or handle not held")

        # === Step 1: rotate gripper to 90 degrees for proper alignment ===
        print("[Task] Rotating gripper from zero_deg to ninety_deg")
        obs, reward, done = rotate(env, task, gripper, zero_deg, ninety_deg)
        if done:
            print("[Task] Task ended after rotate")
            return

        # === Step 2: move gripper to side position of the drawer ===
        print(f"[Task] Moving gripper to side position: {side_pos}")
        # current gripper pose is in obs.gripper_pose[:3]
        current_pos = task.get_observation().gripper_pose[:3]
        obs, reward, done = move_to_side(env, task, gripper, drawer, current_pos, side_pos)
        if done:
            print("[Task] Task ended after move_to_side")
            return

        # === Step 3: move gripper from side to anchor position ===
        print(f"[Task] Moving gripper from side_pos to anchor_pos: {anchor_pos}")
        obs, reward, done = move_to_anchor(env, task, gripper, drawer, side_pos, anchor_pos)
        if done:
            print("[Task] Task ended after move_to_anchor")
            return

        # === Step 4: grasp the drawer handle at the anchor ===
        print("[Task] Picking drawer at anchor position")
        obs, reward, done = pick_drawer(env, task, gripper, drawer, anchor_pos)
        if done:
            print("[Task] Task ended after pick_drawer")
            return

        # === Step 5: pull the drawer open ===
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task, gripper, drawer)
        if done:
            print("[Task] Task ended after pull")
            return

        print("[Task] Drawer opened successfully")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()