import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Identify which objects we will manipulate
        object_keys = [k for k in positions.keys() if k.startswith('object')]
        # Choose a drop location key, fallback if named differently
        drop_loc = positions.get('dropzone',
                   positions.get('tray',
                   positions.get('base', None)))
        if drop_loc is None:
            raise RuntimeError("No drop location specified in positions.")

        # === Execute Plan ===
        for obj_key in object_keys:
            target_pos = positions[obj_key]
            print(f"[Task] Attempting to pick {obj_key} at: {target_pos}")
            # Try to pick, handle "handempty" precondition failures
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=target_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                msg = str(e).lower()
                if 'handempty' in msg:
                    print("[Task] Hand not empty. Placing whatever is held at drop location.")
                    obs, reward, done = place(env, task, target_pos=drop_loc)
                    if done:
                        print("[Task] Task ended during cleanup placement.")
                        return
                    print("[Task] Retrying pick after emptying hand.")
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=target_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                else:
                    # Unknown exception, re-raise
                    raise

            if done:
                print(f"[Task] Task ended after picking {obj_key}!")
                return

            # Place the picked object to the drop location
            print(f"[Task] Placing {obj_key} at drop location: {drop_loc}")
            obs, reward, done = place(env, task, target_pos=drop_loc)
            if done:
                print(f"[Task] Task ended after placing {obj_key}!")
                return

        print("[Task] All objects processed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()