import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset task and wrap step/get_observation for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve all known object positions
        positions = get_object_positions()
        # find side and anchor positions by name convention
        side_keys = [k for k in positions.keys() if 'side' in k.lower()]
        anchor_keys = [k for k in positions.keys() if 'anchor' in k.lower()]
        if not side_keys or not anchor_keys:
            raise RuntimeError("Missing side or anchor positions in get_object_positions()")
        side_pos = np.array(positions[side_keys[0]])
        anchor_pos = np.array(positions[anchor_keys[0]])

        # Step 1: Move gripper to side position of drawer
        print(f"[Task] Moving to drawer side position at {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended unexpectedly after move to side.")
            return

        # Step 2: Rotate gripper so it can grab the handle
        print("[Task] Rotating gripper to ninety_deg orientation")
        # assume rotate signature rotate(env, task, from_angle, to_angle)
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended unexpectedly after rotate.")
            return

        # Step 3: Move gripper from side to anchor position (handle)
        print(f"[Task] Moving to handle (anchor) position at {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended unexpectedly after move to anchor.")
            return

        # Step 4: Pick the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Task] Task ended unexpectedly after pick.")
            return

        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended after pulling the drawer.")
            return

        print("[Task] Drawer should now be open.")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()