import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        if not positions:
            print("No objects found in the scene; aborting.")
            return

        # If there are at least two objects, pick the first and place it at the second
        object_keys = list(positions.keys())
        if len(object_keys) < 2:
            print("Need at least two objects to demonstrate pick & place.")
            return

        obj1_name = object_keys[0]
        obj2_name = object_keys[1]
        obj1_pos = positions[obj1_name]
        obj2_pos = positions[obj2_name]

        # === Oracle Plan Execution ===
        # Step 1: Pick the first object
        print(f"[Task] Picking '{obj1_name}' at position {obj1_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(obj1_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished during pick; exiting.")
            return

        # Step 2: Place it at the second object's location
        print(f"[Task] Placing '{obj1_name}' at position {obj2_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(obj2_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished during place; exiting.")
            return

        print("[Task] Completed pick & place sequence successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()