import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect at least these keys in the positions dictionary:
        #   'tomato1', 'table', 'bowl'
        if not all(k in positions for k in ('tomato1', 'table', 'bowl')):
            raise KeyError("Expected objects 'tomato1', 'table', 'bowl' in positions")

        tomato1_pos = np.array(positions['tomato1'])
        table_pos   = np.array(positions['table'])
        bowl_pos    = np.array(positions['bowl'])

        # === Exploration Phase ===
        # Move to the table so that tomato1 becomes identified/temperature-known/etc.
        print("[Task] Moving to table at:", table_pos)
        try:
            obs, reward, done = move(env, task, table_pos)
        except Exception as e:
            print(f"[Task] Error during move to table: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after moving to table")
            return

        # === Pick Phase ===
        print("[Task] Picking tomato1 at:", tomato1_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Error during pick: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after pick")
            return

        # === Place Phase ===
        print("[Task] Placing tomato1 in bowl at:", bowl_pos)
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=bowl_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Error during place: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after place")
            return

        print("[Task] Plan execution completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
