import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the open‐drawer task in simulation.'''
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions from the scene
        positions = get_object_positions()
        # The dictionary should contain keys like:
        # 'drawer_side_pos', 'drawer_anchor_pos', 'drawer_handle'
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        drawer_handle_pos = positions['drawer_handle']

        # === Oracle Plan Execution ===

        # 1) Move close to the side of the drawer
        print("[Plan] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("[Plan] Task ended prematurely after move to side.")
            return

        # 2) Rotate the gripper from its home angle to ninety degrees
        print("[Plan] Rotating gripper from zero_deg to ninety_deg")
        obs, reward, done = rotate(env, task, from_angle='zero_deg', to_angle='ninety_deg')
        if done:
            print("[Plan] Task ended prematurely after rotate.")
            return

        # 3) Move into the anchor position at the drawer face
        print("[Plan] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Plan] Task ended prematurely after move to anchor.")
            return

        # 4) Pick the drawer handle
        print("[Plan] Picking drawer handle at:", drawer_handle_pos)
        obs, reward, done = pick(env, task, target_pos=drawer_handle_pos)
        if done:
            print("[Plan] Task ended prematurely after pick.")
            return

        # 5) Pull to open the drawer
        print("[Plan] Pulling drawer to open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] Task ended after pull (drawer should now be open).")
            return

        print("[Plan] Drawer opening sequence completed successfully.")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()