import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # get positions from the scene
        positions = get_object_positions()
        # these keys must match object_positions.py
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        
        # define orientation quaternions
        # ninety degrees about y‑axis, for example
        quat_ninety_deg = [0.0, 0.7071, 0.0, 0.7071]
        # zero rotation
        quat_zero = [0.0, 0.0, 0.0, 1.0]
        
        # 1) Rotate gripper so it faces the drawer side
        print("[Task] Rotating gripper to side orientation")
        obs, reward, done = rotate(env, task, quat_ninety_deg,
                                   max_steps=100, threshold=0.02, timeout=5.0)
        if done:
            print("[Task] Early termination after first rotate")
            return
        
        # 2) Move the gripper to the side position of the drawer handle
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos,
                                 approach_distance=0.10,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Task] Early termination after moving to side")
            return
        
        # 3) Rotate gripper back to zero orientation to align with handle
        print("[Task] Rotating gripper to anchor orientation")
        obs, reward, done = rotate(env, task, quat_zero,
                                   max_steps=100, threshold=0.02, timeout=5.0)
        if done:
            print("[Task] Early termination after second rotate")
            return
        
        # 4) Move the gripper to the anchor position (handle) precisely
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='x',
                                 timeout=5.0)
        if done:
            print("[Task] Early termination after moving to anchor")
            return
        
        # 5) Pick (grasp) the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.0,
                                 max_steps=50,
                                 threshold=0.005,
                                 approach_axis='x',
                                 timeout=5.0)
        if done:
            print("[Task] Early termination after pick")
            return
        
        # 6) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task,
                                 target_pos=None,
                                 max_steps=80,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Early termination after pull")
            return
        
        print("[Task] Drawer should now be open.")
        
    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()