import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve scene object positions
        positions = get_object_positions()
        if not positions:
            raise RuntimeError("No object positions returned by get_object_positions()")

        # identify the drawer and its handle positions
        drawer_names = [name for name in positions if "drawer" in name]
        if not drawer_names:
            raise RuntimeError("No drawer object found in positions")
        drawer_name = drawer_names[0]

        # assume side and anchor positions are stored under specific keys
        side_key = f"{drawer_name}_side_pos"
        anchor_key = f"{drawer_name}_anchor_pos"
        if side_key not in positions or anchor_key not in positions:
            raise RuntimeError(f"Expected keys '{side_key}' and '{anchor_key}' in positions")
        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]

        # assume a single gripper object
        gripper_names = [name for name in positions if "gripper" in name]
        if not gripper_names:
            raise RuntimeError("No gripper object found in positions")
        gripper_name = gripper_names[0]

        # 1) Rotate the gripper to ninety degrees
        # change this quaternion if your ninety_deg is defined differently
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to ninety_deg orientation")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Task ended during rotation")
            return

        # 2) Move the gripper to the side position of the drawer
        current_obs = task.get_observation()
        start_pos = current_obs.gripper_pose[:3]
        print(f"[Task] Moving gripper from {start_pos} to drawer side position {side_pos}")
        obs, reward, done = move_to_side(env, task, gripper_name, drawer_name, start_pos, side_pos)
        if done:
            print("[Task] Task ended during move-to-side")
            return

        # 3) Move the gripper from side position to anchor position
        print(f"[Task] Moving gripper from side position {side_pos} to anchor position {anchor_pos}")
        obs, reward, done = move_to_anchor(env, task, gripper_name, drawer_name, side_pos, anchor_pos)
        if done:
            print("[Task] Task ended during move-to-anchor")
            return

        # 4) Pick the drawer handle at the anchor position
        print("[Task] Picking the drawer handle")
        obs, reward, done = pick_drawer(env, task, gripper_name, drawer_name, anchor_pos)
        if done:
            print("[Task] Task ended during pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task, gripper_name, drawer_name)
        if done:
            print("[Task] Task ended during pull")
            return

        print("===== Task Completed: Drawer is open =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()