import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Exploration skeleton: visit each object, gather information, and determine lock-known for drawers.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve positions of all entities, including robot
        positions = get_object_positions()
        # Expect positions to contain a 'robot' key for initial robot location
        robot_loc = positions.get('robot', None)

        if robot_loc is None:
            print("[Error] No 'robot' entry in positions; cannot perform move actions.")
            return

        # Iterate through all named objects in the environment
        for name, pos in positions.items():
            # Skip the robot itself
            if name == 'robot':
                continue

            print(f"[Task] Exploring '{name}' at position {pos}")
            # 1) Move the robot to the object's location
            try:
                obs, reward, done = move(env, task, robot_loc, pos)
            except Exception as e:
                print(f"[Error] move to {name} failed: {e}")
                break
            if done:
                print("[Task] Episode ended during move.")
                break
            robot_loc = pos  # update current robot location

            # 2) If this is a drawer, attempt to pull to learn lock state
            if 'drawer' in name.lower():
                # ensure gripper is holding the drawer handle before pull
                try:
                    obs, reward, done = pick(env, task, name, pos)
                except Exception as e:
                    print(f"[Error] pick drawer handle for '{name}' failed: {e}")
                    break
                if done:
                    print("[Task] Episode ended during drawer pick.")
                    break
                try:
                    obs, reward, done = pull(env, task, name, pos)
                except Exception as e:
                    print(f"[Error] pull on drawer '{name}' failed: {e}")
                    break
                if done:
                    print("[Task] Episode ended during pull.")
                    break
                # return handle
                try:
                    obs, reward, done = place(env, task, name, pos)
                except Exception:
                    pass
                print(f"[Info] Completed lock-state exploration for drawer '{name}'")
                continue

            # 3) For normal objects: pick and learn weight/durability
            try:
                obs, reward, done = pick(env, task, name, pos)
            except Exception as e:
                print(f"[Error] pick object '{name}' failed: {e}")
                break
            if done:
                print("[Task] Episode ended during pick.")
                break

            # 4) Place the object back to its original location
            try:
                obs, reward, done = place(env, task, name, pos)
            except Exception as e:
                print(f"[Warning] place object '{name}' back failed: {e}")
            if done:
                print("[Task] Episode ended during place.")
                break

            print(f"[Info] Completed exploration for object '{name}'")

        print("===== Exploration Completed =====")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()