import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve a dictionary of object positions
        positions = get_object_positions()
        if not positions:
            print("No object positions available, exiting.")
            return

        # identify a disposal container or bin in the scene
        container_key = None
        for name in positions:
            lname = name.lower()
            if "bin" in lname or "container" in lname or "basket" in lname or "trash" in lname:
                container_key = name
                break
        if container_key is None:
            raise ValueError("Could not find a disposal container in object positions")

        container_pos = positions[container_key]
        print(f"[Task] Using container '{container_key}' at {container_pos}")

        # collect keys of all other objects to process
        object_keys = [k for k in positions.keys() if k != container_key]
        if not object_keys:
            print("No other objects to process, exiting.")
            return

        # loop over each object, pick it up and place into the container
        for obj_key in object_keys:
            obj_pos = positions[obj_key]
            print(f"[Task] Manipulating object '{obj_key}' at {obj_pos}")

            # use pick skill to grasp the object
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] pick failed for {obj_key}: {e}")
                return
            if done:
                print("[Task] Task ended during pick!")
                return

            # use place skill to drop the object into the container
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=container_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] place failed for {obj_key}: {e}")
                return
            if done:
                print("[Task] Task ended during place!")
                return

        print("===== Task Plan Completed Successfully =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()