import numpy as np
import time
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions and orientations from object_positions
        positions = get_object_positions()
        # Expected keys in positions:
        #   'zero_deg'     : initial gripper orientation quaternion
        #   'ninety_deg'   : target gripper orientation quaternion for opening
        #   'side_pos'     : a 3-element tuple for gripper side approach
        #   'anchor_pos'   : a 3-element tuple for gripper anchor position
        #   'drawer_handle': (optional) position of drawer handle if separate
        zero_quat = positions.get('zero_deg')
        ninety_quat = positions.get('ninety_deg')
        side_pos = positions.get('side_pos')
        anchor_pos = positions.get('anchor_pos')
        if zero_quat is None or ninety_quat is None or side_pos is None or anchor_pos is None:
            raise KeyError("Missing required keys in positions: 'zero_deg','ninety_deg','side_pos','anchor_pos'")

        # 1) Ensure gripper starts at zero_deg orientation (exploration/identification phase)
        print("[Task] Rotating gripper to known zero_deg orientation for predicate identification")
        obs, reward, done = rotate(env, task, zero_quat)
        if done:
            print("[Task] Episode ended during initial rotate to zero_deg")
            return

        # 2) Rotate gripper to ninety_deg so that we can approach drawer side
        print("[Task] Rotating gripper to ninety_deg for drawer approach")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Episode ended during rotate to ninety_deg")
            return

        # 3) Move gripper to the side position of the drawer
        print(f"[Task] Moving gripper to side position: {side_pos}")
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended during move to side position")
            return

        # 4) Move gripper from side to the anchor (drawer handle) position
        print(f"[Task] Moving gripper to anchor position: {anchor_pos}")
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.015,
            max_steps=100,
            threshold=0.008,
            approach_axis='x',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended during move to anchor position")
            return

        # 5) Pick the drawer handle (pick-drawer equivalent)
        print("[Task] Picking the drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.005,
            max_steps=80,
            threshold=0.005,
            approach_axis='x',
            timeout=5.0
        )
        if done:
            print("[Task] Episode ended during pick of drawer handle")
            return

        # 6) Pull to open the drawer
        print("[Task] Pulling to open the drawer")
        obs, reward, done = pull(
            env,
            task,
            max_steps=50,
            pull_distance=0.1,
            timeout=5.0
        )
        if done:
            print("[Task] Episode ended during pull action")
            return

        print("===== Task completed: drawer should be open =====")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()