import numpy as np  
from pyrep.objects.shape import Shape  
from pyrep.objects.proximity_sensor import ProximitySensor  

from env import setup_environment, shutdown_environment  
from skill_code import move, pick, place, rotate, pull  
from video import init_video_writers, recording_step, recording_get_observation  
from object_positions import get_object_positions  

def run_skeleton_task():  
    print("===== Starting Skeleton Task =====")  

    # === Environment Setup ===  
    env, task = setup_environment()  
    try:  
        descriptions, obs = task.reset()  
        init_video_writers(obs)  

        # wrap step and get_observation for recording  
        original_step = task.step  
        task.step = recording_step(original_step)  
        original_get_obs = task.get_observation  
        task.get_observation = recording_get_observation(original_get_obs)  

        # === Retrieve Object Positions ===  
        positions = get_object_positions()  
        # Example: positions might contain keys 'object_1', 'object_2', 'drawer_side', 'drawer_anchor'  
        if 'object_1' not in positions or 'object_2' not in positions:  
            print("Error: expected 'object_1' and 'object_2' in positions, got:", positions.keys())  
            return  
        if 'drawer_side' not in positions or 'drawer_anchor' not in positions:  
            print("Error: expected 'drawer_side' and 'drawer_anchor' in positions for drawer manipulation")  
            return  

        object_1_pos = positions['object_1']  
        object_2_pos = positions['object_2']  
        drawer_side_pos = positions['drawer_side']  
        drawer_anchor_pos = positions['drawer_anchor']  

        # === Define target quaternions ===  
        # ninety degrees around z  
        sin45, cos45 = np.sin(np.pi/4), np.cos(np.pi/4)  
        quat_ninety = [0.0, 0.0, sin45, cos45]  
        quat_zero = [0.0, 0.0, 0.0, 1.0]  

        # === Oracle Plan ===  
        # This is the sequence of skill calls needed to complete the task  
        # Each entry is (action_name, list_of_positional_args, dict_of_keyword_args)  
        plan = [  
            # rotate gripper to 90° so we can approach the drawer handle properly  
            ("rotate", [quat_ninety], {}),  

            # explore and manipulate object_1  
            ("move", [object_1_pos], {"approach_distance": 0.15, "max_steps": 100, "threshold": 0.01, "approach_axis": "z", "timeout": 10.0}),  
            ("pick", [object_1_pos], {"approach_distance": 0.05, "max_steps": 100, "threshold": 0.01, "approach_axis": "z", "timeout": 10.0}),  
            ("place", [object_1_pos], {"approach_distance": 0.10, "max_steps": 100, "threshold": 0.01, "approach_axis": "z", "timeout": 10.0}),  

            # explore and manipulate object_2  
            ("move", [object_2_pos], {"approach_distance": 0.15, "max_steps": 100, "threshold": 0.01, "approach_axis": "z", "timeout": 10.0}),  
            ("pick", [object_2_pos], {"approach_distance": 0.05, "max_steps": 100, "threshold": 0.01, "approach_axis": "z", "timeout": 10.0}),  
            ("place", [object_2_pos], {"approach_distance": 0.10, "max_steps": 100, "threshold": 0.01, "approach_axis": "z", "timeout": 10.0}),  

            # reposition for drawer manipulation  
            ("rotate", [quat_zero], {}),  
            ("move", [drawer_side_pos], {"approach_distance": 0.10, "max_steps": 100, "threshold": 0.01, "approach_axis": "xy", "timeout": 10.0}),  
            ("move", [drawer_anchor_pos], {"approach_distance": 0.05, "max_steps": 100, "threshold": 0.01, "approach_axis": "xy", "timeout": 10.0}),  

            # pick the drawer and pull it open  
            ("pick", [drawer_anchor_pos], {"approach_distance": 0.02, "max_steps": 100, "threshold": 0.005, "approach_axis": "xy", "timeout": 10.0}),  
            ("pull", [], {}),  
        ]  

        # === Execute the plan ===  
        for step_idx, (action_name, args, kwargs) in enumerate(plan, start=1):  
            print(f"[Plan] Step {step_idx}: {action_name} args={args} kwargs={kwargs}")  
            if not hasattr(globals(), action_name):  
                print(f"Error: skill '{action_name}' is not available")  
                break  
            skill_fn = globals()[action_name]  
            try:  
                obs, reward, done = skill_fn(env, task, *args, **kwargs)  
            except Exception as e:  
                print(f"Exception during '{action_name}': {e}")  
                break  
            if done:  
                print(f"Task ended prematurely after '{action_name}'")  
                break  

    finally:  
        shutdown_environment(env)  

    print("===== End of Skeleton Task =====")  

if __name__ == "__main__":  
    run_skeleton_task()