# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull  # use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        if not positions:
            print("[Error] No objects found in the environment.")
            return

        # We will pick the first object and place it at the second object's location
        object_keys = list(positions.keys())
        if len(object_keys) < 2:
            print("[Error] Need at least two objects to execute the plan.")
            return

        obj1_key = object_keys[0]
        obj2_key = object_keys[1]
        obj1_pos = np.array(positions[obj1_key])
        obj2_pos = np.array(positions[obj2_key])

        # === Plan Execution ===

        # 1) Move to the first object
        print(f"[Task] Moving to {obj1_key} at {obj1_pos}")
        try:
            obs, reward, done = move(env, task, obj1_pos)
        except Exception as e:
            print(f"[Error] move to {obj1_key} failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely after move.")
            return

        # 2) Pick up the first object
        print(f"[Task] Picking up {obj1_key}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=obj1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] pick {obj1_key} failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely after pick.")
            return

        # 3) Move to the placement location (object 2's position)
        print(f"[Task] Moving to placement location at {obj2_pos}")
        try:
            obs, reward, done = move(env, task, obj2_pos)
        except Exception as e:
            print(f"[Error] move to placement location failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely after move.")
            return

        # 4) Place the first object at the second object's location
        print(f"[Task] Placing {obj1_key} at {obj2_key} location")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=obj2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] place {obj1_key} failed: {e}")
            return
        if done:
            print("[Task] Episode ended prematurely after place.")
            return

        print("[Task] Plan executed successfully.")
    
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()