import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve positions of all named things in the scene
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", list(positions.keys()))

        # === ORACLE PLAN ===
        # This example plan opens a drawer and then picks & places an object.

        # 1. Open the drawer:
        #    a) rotate gripper to 90 degrees
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Plan] Task ended during rotate. Exiting early.")
            return

        #    b) move to side position of drawer handle
        #       replace 'drawer_side' with the actual key in your positions dict
        side_key = 'drawer_side'
        if side_key not in positions:
            raise KeyError(f"Expected key '{side_key}' in positions")
        side_pos = positions[side_key]
        obs, reward, done = move(env, task, from_pos=None, to_pos=side_pos)
        if done:
            print("[Plan] Task ended after moving to side. Exiting early.")
            return

        #    c) move to anchor position of drawer handle
        anchor_key = 'drawer_anchor'
        if anchor_key not in positions:
            raise KeyError(f"Expected key '{anchor_key}' in positions")
        anchor_pos = positions[anchor_key]
        obs, reward, done = move(env, task, from_pos=side_pos, to_pos=anchor_pos)
        if done:
            print("[Plan] Task ended after moving to anchor. Exiting early.")
            return

        #    d) pick the drawer handle
        handle_name = 'drawer_handle'
        if handle_name not in positions:
            raise KeyError(f"Expected object '{handle_name}' in positions")
        obs, reward, done = pick(env, task, handle_name, anchor_pos)
        if done:
            print("[Plan] Task ended after pick-drawer. Exiting early.")
            return

        #    e) pull to open the drawer
        obs, reward, done = pull(env, task, handle_name, anchor_pos)
        if done:
            print("[Plan] Task ended after pull. Exiting early.")
            return

        print("[Plan] Drawer is now open.")

        # 2. Pick an object (e.g., 'tomato1') from inside the drawer
        obj_key = 'tomato1'
        if obj_key not in positions:
            raise KeyError(f"Expected object '{obj_key}' in positions")
        obj_pos = positions[obj_key]
        obs, reward, done = move(env, task, from_pos=anchor_pos, to_pos=obj_pos)
        if done:
            print("[Plan] Task ended after moving to object. Exiting early.")
            return
        obs, reward, done = pick(env, task, obj_key, obj_pos)
        if done:
            print("[Plan] Task ended after pick-object. Exiting early.")
            return

        # 3. Place the object to target location (e.g., 'bin')
        bin_key = 'bin'
        if bin_key not in positions:
            raise KeyError(f"Expected target location '{bin_key}' in positions")
        bin_pos = positions[bin_key]
        obs, reward, done = move(env, task, from_pos=obj_pos, to_pos=bin_pos)
        if done:
            print("[Plan] Task ended after moving to bin. Exiting early.")
            return
        obs, reward, done = place(env, task, obj_key, bin_pos)
        if done:
            print("[Plan] Task ended after place-object. Exiting early.")
            return

        print("[Plan] Successfully opened drawer, picked object, and placed it in the bin.")

    except Exception as e:
        print(f"[Error] Exception during execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()