import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve object positions from the scene ===
        positions = get_object_positions()
        print(f"[Info] Retrieved positions: {list(positions.keys())}")

        # === Exploration phase: visit every known position to trigger identification ===
        for name, pos in positions.items():
            print(f"[Exploration] Moving to location '{name}' at {pos}")
            try:
                obs, reward, done = move(env, task, None, pos)
                if done:
                    print("[Exploration] Task ended unexpectedly during exploration.")
                    return
            except Exception as e:
                print(f"[Exploration] Warning: failed to move to {name}: {e}")

        # === Locate the drawer handle side and anchor positions ===
        side_keys = [k for k in positions if "side" in k]
        anchor_keys = [k for k in positions if "anchor" in k]
        if not side_keys or not anchor_keys:
            print("[Error] Could not find both 'side' and 'anchor' positions in object list.")
            return
        side_pos = positions[side_keys[0]]
        anchor_pos = positions[anchor_keys[0]]
        print(f"[Info] Using side_pos='{side_keys[0]}' at {side_pos}, anchor_pos='{anchor_keys[0]}' at {anchor_pos}")

        # === Rotate gripper to 90 degrees orientation ===
        # Represent 90° around Z as quaternion [0,0,sin(45°),cos(45°)]
        target_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print(f"[Task] Rotating gripper to 90° quaternion {target_quat}")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended during rotate.")
            return

        # === Move gripper to side position of drawer ===
        print(f"[Task] Moving gripper to side position {side_pos}")
        obs, reward, done = move(env, task, None, side_pos)
        if done:
            print("[Task] Ended during move-to-side.")
            return

        # === Move gripper from side to anchor position ===
        print(f"[Task] Moving gripper to anchor position {anchor_pos}")
        obs, reward, done = move(env, task, side_pos, anchor_pos)
        if done:
            print("[Task] Ended during move-to-anchor.")
            return

        # === Pick the drawer handle at anchor position ===
        print(f"[Task] Picking drawer handle at {anchor_pos}")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Ended during pick-drawer.")
            return

        # === Pull the drawer open ===
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Ended during pull.")
            return

        print("===== Task Completed Successfully =====")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()