import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, place, pull, move_to_side, move_to_anchor, pick_drawer
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step / get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve positions and orientations for drawer manipulation
        positions = get_object_positions()
        # We expect object_positions to return something like:
        # { 'drawer': { 'side_pos': <xyz>, 'anchor_pos': <xyz>, 'anchor_quat': <xyzw> }, ... }
        if 'drawer' not in positions:
            raise KeyError("object_positions() did not return key 'drawer'")
        drawer_info = positions['drawer']
        side_pos = drawer_info.get('side_pos')
        anchor_pos = drawer_info.get('anchor_pos')
        target_quat = drawer_info.get('anchor_quat')
        if side_pos is None or anchor_pos is None or target_quat is None:
            raise ValueError("drawer_info must contain 'side_pos', 'anchor_pos', and 'anchor_quat'")

        # === Oracle plan to open a drawer ===
        # 1) Orient gripper to the handle
        print("[Task] Rotating wrist to target orientation")
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] done after rotate")
            return

        # 2) Move gripper to the side position of the drawer
        print("[Task] Moving gripper to side position on drawer")
        obs, reward, done = move_to_side(env, task, gripper=None, d='drawer', from_pos=None, to_pos=side_pos)
        if done:
            print("[Task] done after move-to-side")
            return

        # 3) Move gripper from side to anchor position
        print("[Task] Moving gripper to anchor position on drawer")
        obs, reward, done = move_to_anchor(env, task, gripper=None, d='drawer', from_pos=side_pos, to_pos=anchor_pos)
        if done:
            print("[Task] done after move-to-anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Grasping drawer handle")
        obs, reward, done = pick_drawer(env, task, gripper=None, d='drawer', p=anchor_pos)
        if done:
            print("[Task] done after pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task, gripper=None, d='drawer')
        if done:
            print("[Task] done after pull")
            return

        print("[Task] Completed oracle plan for opening drawer")

    except Exception as e:
        print(f"[Task] Error during plan execution: {e}")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()