import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def is_gripper_locked(env):
    """Return True if the gripper appears locked (feedback based)."""
    # Based on feedback: negative openness implies locked
    openness = getattr(env, 'gripper_openness', None)
    if openness is None:
        return False
    return openness < 0.0

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve all known object positions
        positions = get_object_positions()
        # ensure 'waypoint1' is in the dictionary
        if 'waypoint1' not in positions:
            print("[Setup] WARNING: 'waypoint1' not found in positions, adding placeholder at origin")
            positions['waypoint1'] = (0.0, 0.0, 0.0)

        # names of objects we know
        drawer = 'drawer1'
        handle = 'drawer1_handle'
        gripper_name = 'gripper'
        # make sure we have the drawer handle position
        if handle not in positions:
            print(f"[Setup] ERROR: handle '{handle}' missing from positions. Cannot proceed.")
            return

        handle_pos = positions[handle]
        waypoint1_pos = positions['waypoint1']

        # === Exploration Phase: identify missing predicate or lock status ===
        print("[Exploration] Attempting to pull drawer to discover lock state...")
        try:
            # This will fail if required predicates (lock-known) are missing
            obs, reward, done = pull(env, task, gripper_name, drawer)
            print("[Exploration] pull action completed without exception.")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            missing_predicate = str(e).split()[0]
            print(f"[Exploration] Identified missing predicate or unsupported action clause: {missing_predicate}")
            return

        # check if gripper is locked before proceeding to rotate
        if is_gripper_locked(env):
            print("[Safety] Gripper appears locked. Aborting rotation attempt.")
            return

        # === Oracle Plan Execution ===
        # 1) approach the handle
        print("[Plan] Approaching drawer handle at", handle_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=handle_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended prematurely during pick.")
            return

        # 2) rotate gripper to ninety_deg before moving to side
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Plan] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=ninety_deg_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended prematurely during rotate.")
            return

        # 3) pull drawer open
        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task, gripper_name, drawer)
        if done:
            print("[Plan] Task ended prematurely during pull.")
            return

        # 4) retract and place handle back at waypoint1 for safety
        print("[Plan] Placing handle at waypoint1:", waypoint1_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=waypoint1_pos,
            drop_height=0.10,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended prematurely during place.")
            return

        print("===== Oracle Plan Execution Complete: Drawer is open =====")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()