import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # Wrap the task for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Identify the drawer handle or equivalent by name
        handle_key = next((k for k in positions.keys() if 'handle' in k), None)
        if handle_key is None:
            print("[Error] No object with 'handle' in its name found in positions.")
            return
        handle_pos = positions[handle_key]
        
        # === Task Plan ===
        # 1) Rotate the gripper to a 90-degree orientation about its vertical axis
        ninety_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to 90 degrees (quat):", ninety_quat)
        obs, reward, done = rotate(env, task, ninety_quat, max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Terminated after rotate")
            return
        
        # 2) Move the robot's end-effector above the drawer handle
        print(f"[Task] Moving end-effector to handle position at {handle_pos}")
        # The 'move' skill is assumed to accept env, task, target position
        obs, reward, done = move(env, task, None, handle_pos)
        if done:
            print("[Task] Terminated after move")
            return
        
        # 3) Pick the drawer handle
        print("[Task] Picking the handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=handle_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated after pick")
            return
        
        # 4) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Terminated after pull")
            return
        
        print("[Task] Completed all steps without early termination.")
    
    finally:
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()