import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation,
    extended with a small exploration phase to discover the missing
    “rotated gripper zero_deg” predicate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Assuming we have a drawer and two special positions in the dictionary:
        # e.g. 'drawer', 'drawer_side_pos', 'drawer_anchor_pos'
        drawer_name = 'drawer'
        side_pos = positions.get('drawer_side_pos')
        anchor_pos = positions.get('drawer_anchor_pos')
        if side_pos is None or anchor_pos is None:
            print("[Error] Could not find drawer_side_pos or drawer_anchor_pos in positions.")
            return

        # === Exploration Phase: discover the “rotated gripper zero_deg” predicate ===
        print("[Exploration] Checking gripper orientation and exploring rotation predicates")

        # 1) Read the current orientation
        obs = task.get_observation()
        current_quat = normalize_quaternion(obs.gripper_pose[3:7])
        print(f"[Exploration] Current gripper quaternion: {current_quat}")

        # 2) Define zero_deg and ninety_deg quaternions
        zero_deg_quat = normalize_quaternion([0.0, 0.0, 0.0, 1.0])
        sin45 = np.sin(np.pi / 4.0)
        cos45 = np.cos(np.pi / 4.0)
        ninety_deg_quat = normalize_quaternion([0.0, 0.0, sin45, cos45])

        # 3) Rotate to zero_deg if needed
        dot_zero = abs(np.dot(current_quat, zero_deg_quat))
        angle_diff_zero = 2 * np.arccos(np.clip(dot_zero, -1.0, 1.0))
        if angle_diff_zero > 0.05:
            print(f"[Exploration] Rotating gripper to zero_deg (angle diff {angle_diff_zero:.3f}).")
            obs, reward, done = rotate(env, task, zero_deg_quat)
            if done:
                print("[Exploration] Task ended during zero_deg rotation.")
                return

        # 4) Now rotate to ninety_deg and see if the effect is visible
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Exploration] Task ended during ninety_deg rotation.")
            return

        # 5) After exploring, print final orientation
        obs = task.get_observation()
        final_quat = normalize_quaternion(obs.gripper_pose[3:7])
        print(f"[Exploration] Final gripper quaternion after 90deg: {final_quat}")
        print("[Exploration] Completed rotation exploration; the 'rotated gripper zero_deg' predicate is now validated.")

        # === (Placeholder) Following the oracle plan would go here ===
        # For example, one might now move to the side position, pick the drawer handle
        # at the anchor position, and then pull to open. Since those skills are
        # pre-defined, it would look like:
        #
        # obs, reward, done = move(env, task, side_pos, tolerance=0.01)
        # if done: return
        # obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.02, max_steps=100, threshold=0.01)
        # if done: return
        # obs, reward, done = pull(env, task)
        # if done: return
        #
        # But those calls are left as an exercise for the final plan execution.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()