import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']

        # Step 1: Rotate the gripper to 90 degrees
        print("[Task] Rotating to 90 degrees...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # 90-degree rotation
        if done:
            return

        # Step 2: Move the gripper to the side position to approach the drawer
        print("[Task] Moving to the side position...")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            return

        # Step 3: Move the gripper to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position...")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer...")
        obs, reward, done = pick_drawing(env, task, g='gripper', d='bottom', p='anchor-pos-bottom')
        if done:
            return

        # Step 5: Pull the bottom drawer to open it
        print("[Task] Pulling the drawer...")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x') 
        if done:
            return

        # Step 6: Pick up the rubbish from the table
        print("[Task] Picking up the rubbish...")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15)
        if done:
            return

        # Step 7: Place the rubbish into the bin
        print("[Task] Placing rubbish in the bin...")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()