import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define positions for the rubbish and the bin
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']
        
        # Define gripper position for the drawer interaction
        gripper_pos = positions['bottom_anchor_pos']  # Gripper position to interact with the drawer

        # === Start executing the plan ===
        # Step 1: Rotate the gripper to ninety degrees
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]), max_steps=100)
        if done:
            print("[Task] Gripper rotated successfully!")
            
            # Step 2: Move to the side position of the drawer
            print("[Task] Moving to the side position of the drawer.")
            obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'], max_steps=100, threshold=0.01, timeout=10.0)
            if done:
                print("[Task] Moved to side position successfully!")
                
                # Step 3: Move to the anchor position of the drawer
                print("[Task] Moving to the anchor position of the drawer.")
                obs, reward, done = move(env, task, target_pos=gripper_pos, max_steps=100, threshold=0.01, timeout=10.0)
                if done:
                    print("[Task] Moved to anchor position successfully!")

                    # Step 4: Pick the drawer
                    print("[Task] Picking the drawer.")
                    obs, reward, done = pick(env, task, target_pos=gripper_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                    if done:
                        print("[Task] Drawer picked successfully!")

                        # Step 5: Pull to open the drawer
                        print("[Task] Pulling the drawer to open.")
                        obs, reward, done = pull(env, task, pull_distance=0.5, pull_axis='-x', max_steps=100, threshold=0.01, timeout=10.0)
                        if done:
                            print("[Task] Drawer opened successfully!")

                            # Step 6: Move to pick the rubbish
                            print("[Task] Moving to the rubbish position.")
                            obs, reward, done = move(env, task, target_pos=rubbish_pos, max_steps=100, threshold=0.01, timeout=10.0)
                            if done:
                                print("[Task] Moved to rubbish position successfully!")

                                # Step 7: Pick the rubbish
                                print("[Task] Picking up the rubbish.")
                                obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                                if done:
                                    print("[Task] Rubbish picked successfully!")
                                    
                                    # Step 8: Place the rubbish in the bin
                                    print("[Task] Placing rubbish in the bin.")
                                    obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                                    if done:
                                        print("[Task] Rubbish placed in the bin successfully! Goal accomplished.")
                                    else:
                                        print("[Task] Failed to place the rubbish in the bin.")
                                else:
                                    print("[Task] Failed to pick the rubbish.")
                            else:
                                print("[Task] Failed to move to rubbish position.")
                        else:
                            print("[Task] Failed to pull the drawer.")
                    else:
                        print("[Task] Failed to pick the drawer.")
                else:
                    print("[Task] Failed to move to anchor position.")
            else:
                print("[Task] Failed to move to side position.")
        else:
            print("[Task] Failed to rotate the gripper.")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()