import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Retrieve references to the objects in the environment
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_side_pos = positions['bottom_side_pos']

        # === Execute the Plan ===
        # Step 1: Rotate gripper to be aligned for side move
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 0, 1]))  # Assuming zero rotation
        if done:
            print("[Task] Gripper rotated to initial position successfully!")
        else:
            print("[Task] Task not completed yet (done=False).")
            return  # Exit if rotation couldn't be completed

        # Step 2: Move the gripper to the side position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=bottom_side_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Moved to the side position of the bottom drawer successfully!")
        else:
            print("[Task] Task not completed yet (done=False).")
            return  # Exit if unable to move

        # Step 3: Move to the anchor position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Moved to the anchor position of the bottom drawer successfully!")
        else:
            print("[Task] Task not completed yet (done=False).")
            return  # Exit if unable to move

        # Step 4: Pick the bottom drawer
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Picked the bottom drawer successfully!")
        else:
            print("[Task] Task not completed yet (done=False).")
            return  # Exit if unable to pick

        # Step 5: Pull the drawer open
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Drawer opened successfully!")
        else:
            print("[Task] Task not completed yet (done=False).")
            return  # Exit if unable to pull

        # Step 6: Place rubbish into the bin
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Rubbish placed into the bin successfully!")
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()