import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define specific positions for the rubbish and bin.
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        side_pos_bottom = positions['bottom_side_pos']

        # Step 1: Rotate the gripper to 90 degrees
        print("[Task] Rotating gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # Rotate by 90 degrees
        if done:
            print("[Task] Gripper rotated successfully.")
        else:
            print("[Task] Gripper rotation failed.")
            return

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving gripper to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=side_pos_bottom) 
        if done:
            print("[Task] Gripper moved to side position successfully.")
        else:
            print("[Task] Failed to move to the side position.")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving gripper to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos) 
        if done:
            print("[Task] Gripper moved to anchor position successfully.")
        else:
            print("[Task] Failed to move to the anchor position.")
            return
            
        # Step 4: Pick the bottom drawer
        print("[Task] Picking the drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Drawer picked successfully.")
        else:
            print("[Task] Failed to pick the drawer.")
            return

        # Step 5: Pull to open the drawer
        print("[Task] Pulling to open the drawer.")
        obs, reward, done = pull(env, task, pull_distance=0.3, pull_axis='y', max_steps=100)  # Pull action with default params
        if done:
            print("[Task] Drawer opened successfully.")
        else:
            print("[Task] Unable to open drawer.")
            return  

        # Step 6: Pick up the rubbish
        print("[Task] Picking up rubbish.")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15)
        if done:
            print("[Task] Rubbish picked successfully.")
        else:
            print("[Task] Failed to pick up rubbish.")
            return 

        # Step 7: Place rubbish in the bin
        print("[Task] Placing rubbish in the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Rubbish placed successfully.")
        else:
            print("[Task] Failed to place rubbish.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()